#!/usr/bin/env python3
# -*- coding:utf-8 -*-
###
# File: /home/richard/projects/DeepOrchestration/componants.py
# Project: /home/richard/projects/DeepOrchestration/utils
# Created Date: Wednesday, May 29th 2024, 5:06:30 pm
# Author: Ruochi Zhang
# Email: zrc720@gmail.com
# -----
# Last Modified: Thu May 30 2024
# Modified By: Ruochi Zhang
# -----
# Copyright (c) 2024 Bodkin World Domination Enterprises
#
# MIT License
#
# Copyright (c) 2024 Ruochi Zhang
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do
# so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# -----
###

import os
import sys

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from ..dataloader import mimic_dataloader
from ..models.model_loaders import model_preperation
from .trainer import Trainer
from .loss import LossFunction, DeepOrLoss
from .evaluator import Evaluator
from .utils import get_device
from .scheduler import lr_scheduler
from .metrics import BinaryMetrics
from ..logger.mlflow_logger import MlflowLog
from ..logger.std_logger import StdLogger
import torch
import logging


def deepor_componants(cfg, orig_cwd):

    if cfg.logger.mlflow:
        logger = MlflowLog()
    else:
        logger = StdLogger("", level=logging.INFO)

    device = get_device(cfg)
    dataloaders = mimic_dataloader.dataloaders_preperation(cfg, orig_cwd)

    net = model_preperation(cfg)

    logger.std_print("Modeling Done......")
    logger.std_print("arch is {}".format(net))

    optimizer = torch.optim.Adam(net.parameters(),
                                 lr=cfg.train.learning_rate,
                                 weight_decay=cfg.train.weight_decay)
    scheduler = lr_scheduler(cfg, optimizer)

    if cfg.model.name == "deepor":
        criterion = DeepOrLoss(device,
                               alpha=cfg.model.deepor.loss.alpha,
                               beta=cfg.model.deepor.loss.beta,
                               gamma=cfg.model.deepor.loss.gamma)

    elif cfg.model.name == "baseline":
        criterion = LossFunction(device)

    metrics = BinaryMetrics()
    global_rank = 0

    my_trainer = Trainer(net, criterion, dataloaders, optimizer, scheduler,
                         metrics, global_rank, cfg, logger, device)
    # evaluation
    evaluetor = Evaluator(net,
                          test_loader=dataloaders['test'],
                          criterion=criterion,
                          metric_func=metrics,
                          device=device,
                          cfg=cfg)

    return net, my_trainer, evaluetor, logger
